// DEMO BINARY OPTION FRONTEND

const PRICE_API = "https://api.binance.com/api/v3/ticker/price?symbol=";
const SUPPORTED_PAIRS = ["BTCUSDT", "ETHUSDT"];

const DEMO_QUOTE = "USDT";
const DEMO_BASE = "BTC";
const DEMO_START_QUOTE = 10000000;
const PAYOUT_RATE = 0.8;

let balances = {
  [DEMO_QUOTE]: { balance: DEMO_START_QUOTE, locked: 0 },
  [DEMO_BASE]: { balance: 0, locked: 0 }
};

let lastPrice = 9500;
let firstPriceForChange = null;
let orderbook = { bids: [], asks: [] };
let trades = [];

let openPositions = [];
let closedPositions = [];
let allPositions = [];
let nextPositionId = 1;

let currentOrderType = "limit";
let currentBottomTab = "open";
let currentDurationSec = 60;

function fmt(num, decimals = 2) {
  const n = Number(num);
  if (!isFinite(n)) return "0";
  return n.toLocaleString("en-US", {
    minimumFractionDigits: decimals,
    maximumFractionDigits: decimals
  });
}

function fmtTime(d) {
  const date = d instanceof Date ? d : new Date(d);
  return date.toLocaleTimeString("en-GB", { hour12: false });
}

function showToast(msg, type = "success") {
  const toast = document.getElementById("toast");
  if (!toast) return;
  toast.textContent = msg;
  toast.className = "toast " + type;
  void toast.offsetWidth;
  toast.classList.add("show");
  setTimeout(() => toast.classList.remove("show"), 2500);
}

function getBestBid() {
  if (!orderbook.bids.length) return null;
  return [...orderbook.bids].sort((a, b) => b.price - a.price)[0];
}

function getBestAsk() {
  if (!orderbook.asks.length) return null;
  return [...orderbook.asks].sort((a, b) => a.price - b.price)[0];
}

function updatePairSymbol() {
  const select = document.getElementById("pair-select");
  const el = document.getElementById("ui-pair-symbol");
  if (!select || !el) return;
  const val = select.value || "BTCUSDT";
  el.textContent = val.endsWith("USDT")
    ? val.replace("USDT", "/USDT")
    : val;
}

function updateTopStats(price) {
  const elLast = document.getElementById("ui-last-price");
  const elChange = document.getElementById("ui-change");
  const elHigh = document.getElementById("ui-high");
  const elLow = document.getElementById("ui-low");
  const elVol = document.getElementById("ui-volume");
  const obLast = document.getElementById("ob-last-price");
  const obUsd = document.getElementById("ob-last-usd");
  const obChange = document.getElementById("ob-last-change");

  if (!elLast || !elChange || !elHigh || !elLow || !elVol) return;

  const p = Number(price);
  if (!isFinite(p) || p <= 0) return;

  elLast.textContent = fmt(p, 2);
  if (obLast) obLast.textContent = fmt(p, 6);
  if (obUsd) obUsd.textContent = fmt(p, 2);

  if (firstPriceForChange === null) firstPriceForChange = p;
  const change = ((p - firstPriceForChange) / firstPriceForChange) * 100;
  const changeTxt = (change >= 0 ? "+" : "") + change.toFixed(2) + "%";
  elChange.textContent = changeTxt;
  elChange.style.background = change >= 0 ? "#dcfce7" : "#fee2e2";
  elChange.style.color = change >= 0 ? "#166534" : "#b91c1c";

  if (obChange) {
    obChange.textContent = changeTxt;
    obChange.style.color = change >= 0 ? "#16a34a" : "#dc2626";
  }

  const prevHigh = elHigh.dataset.val ? Number(elHigh.dataset.val) : p;
  const prevLow = elLow.dataset.val ? Number(elLow.dataset.val) : p;
  const high = Math.max(prevHigh, p);
  const low = Math.min(prevLow, p);
  elHigh.dataset.val = high;
  elLow.dataset.val = low;
  elHigh.textContent = fmt(high, 2);
  elLow.textContent = fmt(low, 2);

  const vol = trades.reduce((acc, t) => acc + Number(t.amount || 0), 0);
  elVol.textContent = fmt(vol, 6);
}

async function fetchRealPrice(pair) {
  try {
    const res = await fetch(PRICE_API + pair);
    if (!res.ok) throw new Error("HTTP " + res.status);
    const data = await res.json();
    return Number(data.price);
  } catch (e) {
    console.error("Price fetch error", pair, e);
    return null;
  }
}

async function refreshPricesFromMarket() {
  const pairSelect = document.getElementById("pair-select");
  if (!pairSelect) return;
  const currentPair = pairSelect.value || "BTCUSDT";

  const price = await fetchRealPrice(currentPair);
  if (price && price > 0) {
    lastPrice = price;
    buildRandomOrderbook(lastPrice);
    renderOrderbook();
    updateTopStats(lastPrice);

    const priceInput = document.getElementById("price");
    if (priceInput && !priceInput.value) {
      priceInput.value = lastPrice.toFixed(2);
    }
    pushRandomTrade();
    updateVolumePreview();
  }

  const tbody = document.getElementById("market-list-body");
  if (!tbody) return;
  tbody.innerHTML = "";

  for (const p of SUPPORTED_PAIRS) {
    const pr = await fetchRealPrice(p);
    if (!pr) continue;
    const tr = document.createElement("tr");
    const tdPair = document.createElement("td");
    const tdPrice = document.createElement("td");
    const tdChange = document.createElement("td");

    tdPair.textContent = p.replace("USDT", "/USDT");
    tdPrice.textContent = fmt(pr, 2);
    const delta = (Math.random() - 0.5) * 4;
    tdChange.textContent = (delta >= 0 ? "+" : "") + delta.toFixed(2) + "%";
    tdChange.className = delta >= 0 ? "text-green" : "text-red";

    tr.appendChild(tdPair);
    tr.appendChild(tdPrice);
    tr.appendChild(tdChange);
    tbody.appendChild(tr);
  }
}

function buildRandomOrderbook(mid) {
  orderbook.bids = [];
  orderbook.asks = [];
  const steps = 20;
  const stepSize = mid * 0.0015;

  for (let i = 0; i < steps; i++) {
    const bidPrice = mid - stepSize * i;
    const askPrice = mid + stepSize * i;
    const bidAmount = 0.05 + Math.random() * 0.8;
    const askAmount = 0.05 + Math.random() * 0.8;
    orderbook.bids.push({ price: bidPrice, amount: bidAmount });
    orderbook.asks.push({ price: askPrice, amount: askAmount });
  }
  recalcOrderbookTotals();
}

function recalcOrderbookTotals() {
  let cum = 0;
  const sBids = [...orderbook.bids].sort((a, b) => b.price - a.price);
  sBids.forEach((l) => {
    cum += l.amount;
    l.total = cum;
  });
  orderbook.bids = sBids;

  cum = 0;
  const sAsks = [...orderbook.asks].sort((a, b) => a.price - b.price);
  sAsks.forEach((l) => {
    cum += l.amount;
    l.total = cum;
  });
  orderbook.asks = sAsks;
}

function renderOrderbook() {
  const asksBody = document.querySelector("#asks-table tbody");
  const bidsBody = document.querySelector("#bids-table tbody");
  if (!asksBody || !bidsBody) return;

  asksBody.innerHTML = "";
  bidsBody.innerHTML = "";

  const maxAskTotal = orderbook.asks.length
    ? Math.max(...orderbook.asks.map((l) => l.total || l.amount))
    : 1;
  const maxBidTotal = orderbook.bids.length
    ? Math.max(...orderbook.bids.map((l) => l.total || l.amount))
    : 1;

  orderbook.asks.slice(0, 15).forEach((l) => {
    const tr = document.createElement("tr");
    tr.classList.add("ob-ask");
    const tdP = document.createElement("td");
    const tdA = document.createElement("td");
    const tdT = document.createElement("td");
    tdP.textContent = fmt(l.price, 6);
    tdA.textContent = fmt(l.amount, 6);
    tdT.textContent = fmt(l.total, 6);
    tr.appendChild(tdP);
    tr.appendChild(tdA);
    tr.appendChild(tdT);
    const depth = (l.total / maxAskTotal) * 100;
    tr.style.setProperty("--depth", depth.toFixed(2) + "%");
    tr.style.color = "#dc2626";
    asksBody.appendChild(tr);
  });

  orderbook.bids.slice(0, 15).forEach((l) => {
    const tr = document.createElement("tr");
    tr.classList.add("ob-bid");
    const tdP = document.createElement("td");
    const tdA = document.createElement("td");
    const tdT = document.createElement("td");
    tdP.textContent = fmt(l.price, 6);
    tdA.textContent = fmt(l.amount, 6);
    tdT.textContent = fmt(l.total, 6);
    tr.appendChild(tdP);
    tr.appendChild(tdA);
    tr.appendChild(tdT);
    const depth = (l.total / maxBidTotal) * 100;
    tr.style.setProperty("--depth", depth.toFixed(2) + "%");
    tr.style.color = "#16a34a";
    bidsBody.appendChild(tr);
  });
}

function pushRandomTrade() {
  const side = Math.random() > 0.5 ? "buy" : "sell";
  const amount = 0.01 + Math.random() * 0.1;
  trades.unshift({
    time: new Date(),
    side,
    price: lastPrice,
    amount
  });
  if (trades.length > 100) trades.length = 100;
  renderRightTrades();
}

function renderRightTrades() {
  const tbody = document.querySelector("#trades-table tbody");
  if (!tbody) return;
  tbody.innerHTML = "";

  trades.slice(0, 40).forEach((t) => {
    const tr = document.createElement("tr");
    const tdTime = document.createElement("td");
    const tdSide = document.createElement("td");
    const tdPrice = document.createElement("td");
    const tdAmt = document.createElement("td");

    tdTime.textContent = fmtTime(t.time);
    tdSide.textContent = t.side.toUpperCase();
    tdSide.style.color = t.side === "buy" ? "#16a34a" : "#dc2626";
    tdPrice.textContent = fmt(t.price, 2);
    tdAmt.textContent = fmt(t.amount, 8);

    tr.appendChild(tdTime);
    tr.appendChild(tdSide);
    tr.appendChild(tdPrice);
    tr.appendChild(tdAmt);
    tbody.appendChild(tr);
  });
}

function renderRightBalances() {
  const tbody = document.querySelector("#balance-table tbody");
  if (!tbody) return;
  tbody.innerHTML = "";

  Object.keys(balances).forEach((asset) => {
    const row = balances[asset];
    const tr = document.createElement("tr");
    const tdAsset = document.createElement("td");
    const tdBal = document.createElement("td");
    const tdLocked = document.createElement("td");

    tdAsset.textContent = asset;
    const isQuote = asset === DEMO_QUOTE;
    tdBal.textContent = fmt(row.balance, isQuote ? 0 : 8);
    tdLocked.textContent = fmt(row.locked, isQuote ? 0 : 8);

    tr.appendChild(tdAsset);
    tr.appendChild(tdBal);
    tr.appendChild(tdLocked);
    tbody.appendChild(tr);
  });

  renderAvailableLines();
}

function renderAvailableLines() {
  const buyRow = document.querySelector(".buy-available span strong");
  const sellRow = document.querySelector(".sell-available span strong");
  if (buyRow) {
    buyRow.textContent = fmt(balances[DEMO_QUOTE].balance, 0) + " " + DEMO_QUOTE;
  }
  if (sellRow) {
    sellRow.textContent = fmt(balances[DEMO_BASE].balance, 8) + " " + DEMO_BASE;
  }
}

function updateVolumePreview() {
  const priceEl = document.getElementById("price");
  const amountEl = document.getElementById("amount");
  const buyStakeEl = document.getElementById("buy-stake");
  const sellStakeEl = document.getElementById("sell-stake");
  const buyPayoutEl = document.getElementById("buy-payout");
  const sellPayoutEl = document.getElementById("sell-payout");
  const buyFeeEl = document.getElementById("buy-fee");
  const sellFeeEl = document.getElementById("sell-fee");

  if (!priceEl || !amountEl) return;

  const p = Number(priceEl.value || lastPrice);
  const a = Number(amountEl.value || 0);
  const stake = p * a;
  const payoutWin = stake * (1 + PAYOUT_RATE);
  const fee = 0;

  if (buyStakeEl) buyStakeEl.textContent = fmt(stake, 2) + " " + DEMO_QUOTE;
  if (sellStakeEl) sellStakeEl.textContent = fmt(stake, 2) + " " + DEMO_QUOTE;
  if (buyPayoutEl) buyPayoutEl.textContent = fmt(payoutWin, 2) + " " + DEMO_QUOTE;
  if (sellPayoutEl) sellPayoutEl.textContent = fmt(payoutWin, 2) + " " + DEMO_QUOTE;
  if (buyFeeEl) buyFeeEl.textContent = fmt(fee, 2) + " " + DEMO_QUOTE;
  if (sellFeeEl) sellFeeEl.textContent = fmt(fee, 2) + " " + DEMO_QUOTE;
}

function renderBottomTab() {
  const table = document.getElementById("openorders-table");
  if (!table) return;
  const thead = table.querySelector("thead");
  const tbody = table.querySelector("tbody");
  if (!thead || !tbody) return;

  thead.innerHTML = "";
  tbody.innerHTML = "";

  if (currentBottomTab === "balance") {
    const hr = document.createElement("tr");
    ["Asset", "Free", "Locked", "Total"].forEach((h) => {
      const th = document.createElement("th");
      th.textContent = h;
      hr.appendChild(th);
    });
    thead.appendChild(hr);

    Object.keys(balances).forEach((asset) => {
      const row = balances[asset];
      const tr = document.createElement("tr");
      const isQuote = asset === DEMO_QUOTE;
      const total = row.balance + row.locked;
      [asset,
       fmt(row.balance, isQuote ? 0 : 8),
       fmt(row.locked, isQuote ? 0 : 8),
       fmt(total, isQuote ? 0 : 8)
      ].forEach((v) => {
        const td = document.createElement("td");
        td.textContent = v;
        tr.appendChild(td);
      });
      tbody.appendChild(tr);
    });
    return;
  }

  const hr = document.createElement("tr");
  ["Time","Pair","Type","Buy/Sell","Entry","Stake","Result","Payout"].forEach((h) => {
    const th = document.createElement("th");
    th.textContent = h;
    hr.appendChild(th);
  });
  thead.appendChild(hr);

  let src = [];
  if (currentBottomTab === "open") src = openPositions;
  else if (currentBottomTab === "closed") src = closedPositions;
  else if (currentBottomTab === "history") src = allPositions;

  src.forEach((pos) => {
    const tr = document.createElement("tr");
    const cols = [
      fmtTime(pos.openedAt),
      pos.pair,
      pos.typeLabel,
      pos.side.toUpperCase(),
      fmt(pos.entryPrice, 2),
      fmt(pos.stake, 2),
      pos.status === "open" ? "OPEN" : pos.result,
      fmt(pos.payout, 2)
    ];
    cols.forEach((v, idx) => {
      const td = document.createElement("td");
      td.textContent = v;
      if (idx === 3) {
        td.style.color = pos.side === "buy" ? "#16a34a" : "#dc2626";
      }
      tr.appendChild(td);
    });
    tbody.appendChild(tr);
  });
}

function initBottomTabs() {
  const tabs = document.querySelectorAll(".bottom-tabs .tab");
  if (!tabs.length) return;
  tabs.forEach((btn) => {
    btn.addEventListener("click", () => {
      tabs.forEach((b) => b.classList.remove("active"));
      btn.classList.add("active");
      const label = btn.textContent.trim().toLowerCase();
      if (label.startsWith("open")) currentBottomTab = "open";
      else if (label.startsWith("closed")) currentBottomTab = "closed";
      else if (label.startsWith("order history")) currentBottomTab = "history";
      else if (label.startsWith("balance")) currentBottomTab = "balance";
      renderBottomTab();
    });
  });
}

function initOrderTypeTabs() {
  const tabs = document.querySelectorAll(".order-tab");
  if (!tabs.length) return;
  tabs.forEach((btn) => {
    btn.addEventListener("click", () => {
      tabs.forEach((b) => b.classList.remove("active"));
      btn.classList.add("active");
      const label = btn.textContent.trim().toLowerCase();
      if (label === "limit") currentOrderType = "limit";
      else if (label === "market") currentOrderType = "market";
      else if (label === "stop limit") currentOrderType = "stopLimit";
      else if (label === "stop market") currentOrderType = "stopMarket";
    });
  });
}

function initDurationButtons() {
  const btns = document.querySelectorAll(".duration-btn");
  if (!btns.length) return;
  btns.forEach((btn) => {
    btn.addEventListener("click", () => {
      btns.forEach((b) => b.classList.remove("active"));
      btn.classList.add("active");
      currentDurationSec = Number(btn.dataset.sec || "0");
    });
  });
}

function initPercentButtons() {
  const percentButtons = document.querySelectorAll(".percent-btn");
  if (!percentButtons.length) return;
  percentButtons.forEach((btn) => {
    btn.addEventListener("click", () => {
      const pct = Number(btn.textContent.replace("%", ""));
      if (!pct) return;

      const priceInput = document.getElementById("price");
      const amountInput = document.getElementById("amount");
      if (!priceInput || !amountInput) return;

      let price = Number(priceInput.value);
      if (!isFinite(price) || price <= 0) {
        const best = getBestAsk() || getBestBid();
        if (best) price = best.price;
      }
      if (!isFinite(price) || price <= 0) return;

      const freeQuote = balances[DEMO_QUOTE].balance;
      const stakeQuote = (freeQuote * pct) / 100;
      const qty = stakeQuote / price;
      amountInput.value = qty > 0 ? qty.toFixed(8) : "";

      updateVolumePreview();
    });
  });
}

function openBinaryPosition(side, rawPrice, rawAmount) {
  const now = new Date();
  const amountNum = Number(rawAmount);
  if (!isFinite(amountNum) || amountNum <= 0) {
    showToast("Amount harus > 0", "error");
    return;
  }

  const pairSelect = document.getElementById("pair-select");
  const pairVal = pairSelect ? pairSelect.value : "BTCUSDT";
  const pairDisp = pairVal.endsWith("USDT")
    ? pairVal.replace("USDT", "/USDT")
    : pairVal;

  let entryPrice = Number(rawPrice);
  if (!isFinite(entryPrice) || entryPrice <= 0) entryPrice = lastPrice;

  const stake = entryPrice * amountNum;
  if (balances[DEMO_QUOTE].balance < stake) {
    showToast("Saldo demo tidak cukup untuk stake ini", "error");
    return;
  }

  balances[DEMO_QUOTE].balance -= stake;

  const pos = {
    id: nextPositionId++,
    pair: pairDisp,
    side,
    type: currentOrderType,
    typeLabel:
      currentOrderType === "limit"
        ? "Limit"
        : currentOrderType === "market"
        ? "Market"
        : currentOrderType === "stopLimit"
        ? "Stop Limit"
        : "Stop Market",
    entryPrice,
    stake,
    openedAt: now,
    expiresAt: currentDurationSec ? now.getTime() + currentDurationSec * 1000 : null,
    durationSec: currentDurationSec,
    status: "open",
    result: "OPEN",
    payout: 0,
    closePrice: null,
    closedAt: null
  };

  openPositions.unshift(pos);
  allPositions.unshift(pos);

  renderRightBalances();
  renderBottomTab();
  updateVolumePreview();

  const resultBox = document.getElementById("order-result");
  if (resultBox) {
    resultBox.textContent = JSON.stringify(
      {
        mode: "binary-demo",
        position: {
          id: pos.id,
          side: pos.side,
          type: pos.typeLabel,
          entryPrice: pos.entryPrice,
          stake: pos.stake,
          durationSec: pos.durationSec,
          expiresAt: pos.expiresAt
        },
        balances
      },
      null,
      2
    );
  }

  showToast(
    (side === "buy" ? "UP" : "DOWN") +
      " dibuka (demo), expiry " +
      (currentDurationSec ? currentDurationSec + "s" : "GTC"),
    "success"
  );
}

function expireCheckLoop() {
  if (!openPositions.length) return;
  const nowTs = Date.now();
  const stillOpen = [];

  openPositions.forEach((pos) => {
    if (pos.expiresAt && pos.expiresAt <= nowTs) {
      const finalPrice = lastPrice;
      const isWin =
        (pos.side === "buy" && finalPrice > pos.entryPrice) ||
        (pos.side === "sell" && finalPrice < pos.entryPrice);

      let payout = 0;
      let result = "LOSE";
      if (isWin) {
        payout = pos.stake * (1 + PAYOUT_RATE);
        result = "WIN";
      }

      balances[DEMO_QUOTE].balance += payout;

      pos.status = "closed";
      pos.result = result;
      pos.payout = payout;
      pos.closePrice = finalPrice;
      pos.closedAt = new Date();

      closedPositions.unshift(pos);
    } else {
      stillOpen.push(pos);
    }
  });

  openPositions = stillOpen;
  renderRightBalances();
  renderBottomTab();
}

function initOrderForm() {
  const form = document.getElementById("order-form");
  if (!form) return;
  const priceEl = document.getElementById("price");
  const amountEl = document.getElementById("amount");

  form.addEventListener("submit", (e) => {
    e.preventDefault();
    const sideInput = document.getElementById("side");
    if (!sideInput || !priceEl || !amountEl) return;
    const side = sideInput.value || "buy";
    const price = priceEl.value || lastPrice;
    const amount = amountEl.value;
    openBinaryPosition(side, price, amount);
    amountEl.value = "";
    updateVolumePreview();
  });

  if (priceEl) priceEl.addEventListener("input", updateVolumePreview);
  if (amountEl) amountEl.addEventListener("input", updateVolumePreview);
}

document.addEventListener("DOMContentLoaded", () => {
  const wsStatus = document.getElementById("ws-status");
  if (wsStatus) {
    wsStatus.textContent = "DEMO";
    wsStatus.className = "badge badge-green";
  }
  const modeLabel = document.getElementById("account-mode-label");
  if (modeLabel) modeLabel.textContent = "Demo Binary Option (tanpa login)";

  const btnLogin = document.getElementById("btn-login");
  const btnRegister = document.getElementById("btn-register");
  if (btnLogin) {
    btnLogin.addEventListener("click", () =>
      showToast("Mode REAL belum dihubungkan (saat ini DEMO).", "error")
    );
  }
  if (btnRegister) {
    btnRegister.addEventListener("click", () =>
      showToast("Pendaftaran REAL belum diaktifkan. Trading tetap DEMO.", "error")
    );
  }

  updatePairSymbol();
  const pairSelect = document.getElementById("pair-select");
  if (pairSelect) {
    pairSelect.addEventListener("change", () => {
      updatePairSymbol();
      refreshPricesFromMarket();
    });
  }

  buildRandomOrderbook(lastPrice);
  renderOrderbook();
  renderRightBalances();
  renderRightTrades();
  updateTopStats(lastPrice);
  updateVolumePreview();

  initOrderTypeTabs();
  initDurationButtons();
  initPercentButtons();
  initBottomTabs();
  initOrderForm();
  renderBottomTab();

  refreshPricesFromMarket();
  setInterval(refreshPricesFromMarket, 2000);
  setInterval(expireCheckLoop, 1000);
});
